<?php
/**
 * Base Request Class holding common functionality for Request Types.
 */
class Request
{
		protected static function GetRequestHash($plainString)
		{
				return md5($plainString);
		}

		protected static function GetFormattedDate()
		{
				return date('d-m-Y:H:i:s:000');	
		}

		protected static function SendRequest($requestString, $testMode)
		{
				if ($testMode){
						$serverUrl = "http://megatron.worldnettps.com/merchant/xmlpayment";
						$serverUrl = "https://testpayments.worldnettps.com/merchant/xmlpayment";
						$serverUrl = "http://localhost:8080/merchant/xmlpayment";
				}
				else
						$serverUrl = "https://payments.worldnettps.com/merchant/xmlpayment";
				
				// Initialisation
				$ch=curl_init();
				// Set parameters
				curl_setopt($ch, CURLOPT_URL, $serverUrl);
				// Return a variable instead of posting it directly
				curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
				// Activate the POST method
				curl_setopt($ch, CURLOPT_POST, 1);
				// Request
				curl_setopt($ch, CURLOPT_POSTFIELDS, $requestString);
				curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
				curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, FALSE);
				// execute the connection
				$result = curl_exec($ch);
				// Close it
				curl_close($ch);
				return $result;
		}

}

/**
 *  Used for processing XML Authorisations through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlAuthRequest extends Request
{
		private $terminalId;
		private $orderId;
		private $currency;
		private $amount;
		public function Amount()
		{
				return $this->amount;
		}
		private $dateTime;
		private $hash;
		private $autoReady;
		private $description;
		private $email;
		private $cardNumber;
		private $cardType;
		private $cardExpiry;
		private $cardHolderName;
		private $cvv;
		private $issueNo;
		private $address1;
		private $address2;
		private $postCode;
		private $cardCurrency;
		private $cardAmount;
		private $conversionRate;
		private $avsOnly;
        private $mpiRef;
		private $multicur = false;

		/**
		 *  Creates the standard request less optional parameters for processing an XML Transaction
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 *  @param orderId A unique merchant identifier. Alpha numeric and max size 12 chars.
		 *  @param currency ISO 4217 3 Digit Currency Code, e.g. EUR / USD / GBP
		 *  @param amount Transaction Amount, Double formatted to 2 decimal places.
		 *  @param description Transaction Description
		 *  @param email Cardholder e-mail
		 *  @param cardNumber A valid Card Number that passes the Luhn Check.
		 *  @param cardType
		 *  Card Type (Accepted Card Types must be configured in the Merchant Selfcare System.)
		 *
		 *  Accepted Values :
		 *
		 *  VISA
		 *  MASTERCARD
		 *  LASER
		 *  SWITCH
		 *  SOLO
		 *  AMEX
		 *  DINERS
		 *  MAESTRO
		 *  DELTA
		 *  ELECTRON
		 *
		 *  @param cardExpiry Card Expiry formatted MMYY
		 *  @param cardHolderName Card Holder Name
		 */
		public function XmlAuthRequest($terminalId,
				$orderId,
				$currency,
				$amount,
				$cardNumber,
				$cardType)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->terminalId = $terminalId;
				$this->orderId = $orderId;
				$this->currency = $currency;
				$this->amount = $amount;
				$this->cardNumber = $cardNumber;
				$this->cardType = $cardType;
		}
	   /**
		 *  Setter for Auto Ready Value
		 *
		 *  @param autoReady
		 *  Auto Ready is an optional parameter and defines if the transaction should be settled automatically.
		 *
		 *  Accepted Values :
		 *
		 *  Y   -   Transaction will be settled in next batch
		 *  N   -   Transaction will not be settled until user changes state in Merchant Selfcare Section
		 */
		public function SetAutoReady($autoReady)
		{
				$this->autoReady = $autoReady;
		}
	   /**
		 *  Setter for Email Address Value
		 *
		 *  @param email Alpha-numeric field.
		 */
		public function SetEmail($email)
		{
				$this->email = $email;
		}
	   /**
		 *  Setter for Email Address Value
		 *
		 *  @param email Alpha-numeric field.
		 */
		public function SetDescription($description)
		{
				$this->description = $description;
		}
	   /**
		 *  Setter for Card Expiry and Card Holder Name values
		 *  These are mandatory for non-SecureCard transactions
		 *
		 *  @param email Alpha-numeric field.
		 */
		public function SetNonSecureCardCardInfo($cardExpiry, $cardHolderName)
		{
				$this->cardExpiry = $cardExpiry;
				$this->cardHolderName = $cardHolderName;
		}
	   /**
		 *  Setter for Card Verification Value
		 *
		 *  @param cvv Numeric field with a max of 4 characters.
		 */
		public function SetCvv($cvv)
		{
				$this->cvv = $cvv;
		}

	   /**
		 *  Setter for Issue No
		 *
		 *  @param issueNo Numeric field with a max of 3 characters.
		 */
		public function SetIssueNo($issueNo)
		{
				$this->issueNo = $issueNo;
		}

	   /**
		 *  Setter for Address Verification Values
		 *
		 *  @param address1 First Line of address - Max size 20
		 *  @param address2 Second Line of address - Max size 20
		 *  @param postCode Postcode - Max size 9
		 */
		public function SetAvs($address1, $address2, $postCode)
		{
				$this->address1 = $address1;
				$this->address2 = $address2;
				$this->postCode = $postCode;
		}
	   /**
		 *  Setter for Foreign Currency Information
		 *
		 *  @param cardCurrency ISO 4217 3 Digit Currency Code, e.g. EUR / USD / GBP
		 *  @param cardAmount (Amount X Conversion rate) Formatted to two decimal places
		 *  @param conversionRate Converstion rate supplied in rate response
		 */
		public function SetForeignCurrencyInformation($cardCurrency, $cardAmount, $conversionRate)
		{
				$this->cardCurrency = $cardCurrency;
				$this->cardAmount = $cardAmount;
				$this->conversionRate = $conversionRate;
		}
	   /**
		 *  Setter for AVS only flag
		 *
		 *  @param avsOnly Only perform an AVS check, do not store as a transaction. Possible values: "Y", "N" 
		 */
		public function SetAvsOnly($avsOnly)
		{
				$this->avsOnly = $avsOnly;
		}
	   /**
		 *  Setter for MPI Reference code
		 *
		 *  @param mpiRef MPI Reference code supplied by WorldNet TPS MPI redirect 
		 */
		public function SetMpiRef($mpiRef)
		{
				$this->mpiRef = $mpiRef;
		}
	   /**
		 *  Setter for multi-currency value
		 *  This is required to be set for multi-currency terminals because the Hash is calculated differently.
		 */
		public function SetMultiCur()
		{
				$this->multicur = true;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlAuthResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				if($this->multicur)
				{
					$this->hash = $this->GetRequestHash($this->terminalId . $this->orderId . $this->currency . $this->amount . $this->dateTime . $sharedSecret);
				} else {
					$this->hash = $this->GetRequestHash($this->terminalId . $this->orderId . $this->amount . $this->dateTime . $sharedSecret);
				}
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlAuthResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("PAYMENT");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("ORDERID");
				$node->appendChild($requestXml->createTextNode($this->orderId));
				$requestString->appendChild($node);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("AMOUNT");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->amount);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CARDNUMBER");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardNumber);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CARDTYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardType);
				$node->appendChild($nodeText);

				if($this->cardExpiry !== NULL && $this->cardHolderName !== NULL)
				{
					$node = $requestXml->createElement("CARDEXPIRY");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->cardExpiry);
					$node->appendChild($nodeText);

					$node = $requestXml->createElement("CARDHOLDERNAME");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->cardHolderName);
					$node->appendChild($nodeText);
				}

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CURRENCY");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->currency);
				$node->appendChild($nodeText);

				if($this->cardCurrency !== NULL && $this->cardAmount > 0 && $this->conversionRate > 0)
				{
					$dcNode = $requestXml->createElement("FOREIGNCURRENCYINFORMATION");
					$requestString->appendChild($dcNode );

					$dcSubNode = $requestXml->createElement("CARDCURRENCY");
					$dcSubNode ->appendChild($requestXml->createTextNode($this->cardCurrency));
					$dcNode->appendChild($dcSubNode);

					$dcSubNode = $requestXml->createElement("CARDAMOUNT");
					$dcSubNode ->appendChild($requestXml->createTextNode($this->cardAmount));
					$dcNode->appendChild($dcSubNode);

					$dcSubNode = $requestXml->createElement("CONVERSIONRATE");
					$dcSubNode ->appendChild($requestXml->createTextNode($this->conversionRate));
					$dcNode->appendChild($dcSubNode);
				}

				$node = $requestXml->createElement("TERMINALTYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode('2');
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TRANSACTIONTYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode('7');
				$node->appendChild($nodeText);

				if($this->autoReady !== NULL)
				{
					$node = $requestXml->createElement("AUTOREADY");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->autoReady);
					$node->appendChild($nodeText);
				}

				if($this->email !== NULL)
				{
					$node = $requestXml->createElement("EMAIL");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->email);
					$node->appendChild($nodeText);
				}

				if($this->cvv !== NULL)
				{
					$node = $requestXml->createElement("CVV");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->cvv);
					$node->appendChild($nodeText);
				}

				if($this->issueNo !== NULL)
				{
					$node = $requestXml->createElement("ISSUENO");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->issueNo);
					$node->appendChild($nodeText);
				}

				if($this->address1 !== NULL  &&$this->address2 !== NULL  &&$this->postCode !== NULL)
				{
					$node = $requestXml->createElement("ADDRESS1");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->address1);
					$node->appendChild($nodeText);

					$node = $requestXml->createElement("ADDRESS2");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->address2);
					$node->appendChild($nodeText);

					$node = $requestXml->createElement("POSTCODE");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->postCode);
					$node->appendChild($nodeText);
				}

				if($this->avsOnly !== NULL)
				{
					$node = $requestXml->createElement("AVSONLY");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->avsOnly);
					$node->appendChild($nodeText);
				}

				if($this->description !== NULL)
				{
					$node = $requestXml->createElement("DESCRIPTION");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->description);
					$node->appendChild($nodeText);
				}

				if($this->mpiRef !== NULL)
				{
					$node = $requestXml->createElement("MPIREF");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->mpiRef);
					$node->appendChild($nodeText);
				}

                return $requestXml->saveXML();

		}
}

/**
 *  Used for processing XML Refund Authorisations through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation. There are no coptional fields.
 */
class XmlRefundRequest extends Request
{
		private $terminalId;
		private $orderId;
		private $amount;
		public function Amount()
		{
				return $this->amount;
		}
		private $dateTime;
		private $hash;
		private $operator;
		private $reason;

		/**
		 *  Creates the refund request for processing an XML Transaction
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 *  @param orderId A unique merchant identifier. Alpha numeric and max size 12 chars.
		 *  @param currency ISO 4217 3 Digit Currency Code, e.g. EUR / USD / GBP
		 *  @param amount Transaction Amount, Double formatted to 2 decimal places.
		 *  @param operator An identifier for who executed this transaction
		 *  @param reason The reason for the refund
		 */
		public function XmlRefundRequest($terminalId,
				$orderId,
				$amount,
				$operator,
				$reason)
		{
				$this->dateTime = $this->GetFormattedDate();
                $this->amount = $amount;
				$this->terminalId = $terminalId;
				$this->orderId = $orderId;
				$this->operator = $operator;
				$this->reason = $reason;
		}
		public function ProcessRequest($sharedSecret, $testMode)
		{
				$this->hash = $this->GetRequestHash($this->terminalId . $this->orderId . $this->amount . $this->dateTime . $sharedSecret);
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlRefundResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("REFUND");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("ORDERID");
				$node->appendChild($requestXml->createTextNode($this->orderId));
				$requestString->appendChild($node);

				$node = $requestXml->createElement("TERMINALID");
				$node->appendChild($requestXml->createTextNode($this->terminalId));
				$requestString->appendChild($node);

				$node = $requestXml->createElement("AMOUNT");
				$node->appendChild($requestXml->createTextNode($this->amount));
				$requestString->appendChild($node);

				$node = $requestXml->createElement("DATETIME");
				$node->appendChild($requestXml->createTextNode($this->dateTime));
				$requestString->appendChild($node);

				$node = $requestXml->createElement("HASH");
				$node->appendChild($requestXml->createTextNode($this->hash));
				$requestString->appendChild($node);

				$node = $requestXml->createElement("OPERATOR");
				$node->appendChild($requestXml->createTextNode($this->operator));
				$requestString->appendChild($node);

				$node = $requestXml->createElement("REASON");
				$node->appendChild($requestXml->createTextNode($this->reason));
				$requestString->appendChild($node);

				return $requestXml->saveXML();

		}
}

/**
 *  Used for processing XML Pre-Authorisations through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlPreAuthRequest extends Request
{
		private $terminalId;
		private $orderId;
		private $currency;
		private $amount;
		public function Amount()
		{
				return $this->amount;
		}
		private $dateTime;
		private $hash;
		private $description;
		private $email;
		private $cardNumber;
		private $cardType;
		private $cardExpiry;
		private $cardHolderName;
		private $cvv;
		private $issueNo;
		private $address1;
		private $address2;
		private $postCode;
		private $cardCurrency;
		private $cardAmount;
		private $conversionRate;
		private $multicur = false;

		/**
		 *  Creates the pre-auth request less optional parameters for processing an XML Transaction
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 *  @param orderId A unique merchant identifier. Alpha numeric and max size 12 chars.
		 *  @param currency ISO 4217 3 Digit Currency Code, e.g. EUR / USD / GBP
		 *  @param amount Transaction Amount, Double formatted to 2 decimal places.
		 *  @param description Transaction Description
		 *  @param email Cardholder e-mail
		 *  @param cardNumber A valid Card Number that passes the Luhn Check.
		 *  @param cardType
		 *  Card Type (Accepted Card Types must be configured in the Merchant Selfcare System.)
		 *
		 *  Accepted Values :
		 *
		 *  VISA
		 *  MASTERCARD
		 *  LASER
		 *  SWITCH
		 *  SOLO
		 *  AMEX
		 *  DINERS
		 *  MAESTRO
		 *  DELTA
		 *  ELECTRON
		 *
		 *  @param cardExpiry Card Expiry formatted MMYY
		 *  @param cardHolderName Card Holder Name
		 */
		public function XmlPreAuthRequest($terminalId,
				$orderId,
				$currency,
				$amount,
				$cardNumber,
				$cardType)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->terminalId = $terminalId;
				$this->orderId = $orderId;
				$this->currency = $currency;
				$this->amount = $amount;
				$this->cardNumber = $cardNumber;
				$this->cardType = $cardType;
		}
	   /**
		 *  Setter for Card Verification Value
		 *
		 *  @param cvv Numeric field with a max of 4 characters.
		 */
		public function SetCvv($cvv)
		{
				$this->cvv = $cvv;
		}

	   /**
		 *  Setter for Email Address Value
		 *
		 *  @param email Alpha-numeric field.
		 */
		public function SetEmail($email)
		{
				$this->email = $email;
		}
	   /**
		 *  Setter for Email Address Value
		 *
		 *  @param email Alpha-numeric field.
		 */
		public function SetDescription($description)
		{
				$this->description = $description;
		}
	   /**
		 *  Setter for Card Expiry and Card Holder Name values
		 *  These are mandatory for non-SecureCard transactions
		 *
		 *  @param email Alpha-numeric field.
		 */
		public function SetNonSecureCardCardInfo($cardExpiry, $cardHolderName)
		{
				$this->cardExpiry = $cardExpiry;
				$this->cardHolderName = $cardHolderName;
		}
	   /**
		 *  Setter for Issue No
		 *
		 *  @param issueNo Numeric field with a max of 3 characters.
		 */
		public function SetIssueNo($issueNo)
		{
				$this->issueNo = $issueNo;
		}

	   /**
		 *  Setter for Address Verification Values
		 *
		 *  @param address1 First Line of address - Max size 20
		 *  @param address2 Second Line of address - Max size 20
		 *  @param postCode Postcode - Max size 9
		 */
		public function SetAvs($address1, $address2, $postCode)
		{
				$this->address1 = $address1;
				$this->address2 = $address2;
				$this->postCode = $postCode;
		}
	   /**
		 *  Setter for Foreign Currency Information
		 *
		 *  @param cardCurrency ISO 4217 3 Digit Currency Code, e.g. EUR / USD / GBP
		 *  @param cardAmount (Amount X Conversion rate) Formatted to two decimal places
		 *  @param conversionRate Converstion rate supplied in rate response
		 */
		public function SetForeignCurrencyInformation($cardCurrency, $cardAmount, $conversionRate)
		{
				$this->cardCurrency = $cardCurrency;
				$this->cardAmount = $cardAmount;
				$this->conversionRate = $conversionRate;
		}
	   /**
		 *  Setter for Multicurrency value
		 */
		public function SetMultiCur()
		{
				$this->multicur = true;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlPreAuthResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				if($this->multicur)
				{
					$this->hash = $this->GetRequestHash($this->terminalId . $this->orderId . $this->currency . $this->amount . $this->dateTime . $sharedSecret);
				} else {
					$this->hash = $this->GetRequestHash($this->terminalId . $this->orderId . $this->amount . $this->dateTime . $sharedSecret);
				}
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlPreAuthResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("PREAUTH");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("ORDERID");
				$node->appendChild($requestXml->createTextNode($this->orderId));
				$requestString->appendChild($node);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("AMOUNT");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->amount);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CARDNUMBER");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardNumber);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CARDTYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardType);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CARDEXPIRY");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardExpiry);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CARDHOLDERNAME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardHolderName);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CURRENCY");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->currency);
				$node->appendChild($nodeText);

				if($this->cardCurrency !== NULL && $this->cardAmount > 0 && $this->conversionRate > 0)
				{
					$dcNode = $requestXml->createElement("FOREIGNCURRENCYINFORMATION");
					$requestString->appendChild($dcNode );

					$dcSubNode = $requestXml->createElement("CARDCURRENCY");
					$dcSubNode ->appendChild($requestXml->createTextNode($this->cardCurrency));
					$dcNode->appendChild($dcSubNode);

					$dcSubNode = $requestXml->createElement("CARDAMOUNT");
					$dcSubNode ->appendChild($requestXml->createTextNode($this->cardAmount));
					$dcNode->appendChild($dcSubNode);

					$dcSubNode = $requestXml->createElement("CONVERSIONRATE");
					$dcSubNode ->appendChild($requestXml->createTextNode($this->conversionRate));
					$dcNode->appendChild($dcSubNode);
				}

				$node = $requestXml->createElement("TERMINALTYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode('2');
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TRANSACTIONTYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode('7');
				$node->appendChild($nodeText);

				if($this->email !== NULL)
				{
					$node = $requestXml->createElement("EMAIL");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->email);
					$node->appendChild($nodeText);
				}

				if($this->cvv !== NULL)
				{
					$node = $requestXml->createElement("CVV");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->cvv);
					$node->appendChild($nodeText);
				}

				if($this->issueNo !== NULL)
				{
					$node = $requestXml->createElement("ISSUENO");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->issueNo);
					$node->appendChild($nodeText);
				}

				if($this->address1 !== NULL  &&$this->address2 !== NULL  &&$this->postCode !== NULL)
				{
					$node = $requestXml->createElement("ADDRESS1");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->address1);
					$node->appendChild($nodeText);

					$node = $requestXml->createElement("ADDRESS2");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->address2);
					$node->appendChild($nodeText);

					$node = $requestXml->createElement("POSTCODE");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->postCode);
					$node->appendChild($nodeText);
				}

				if($this->description !== NULL)
				{
					$node = $requestXml->createElement("DESCRIPTION");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->description);
					$node->appendChild($nodeText);
				}

				return $requestXml->saveXML();

		}
}

/**
 *  Used for processing XML PreAuthorisation Completions through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlPreAuthCompletionRequest extends Request
{
		private $terminalId;
		private $orderId;
		private $amount;
		public function Amount()
		{
				return $this->amount;
		}
		private $dateTime;
		private $hash;
		private $description;
		private $cvv;
		private $cardCurrency;
		private $cardAmount;
		private $conversionRate;
		private $multicur = false;

		/**
		 *  Creates the standard request less optional parameters for processing an XML Transaction
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 *  @param orderId A unique merchant identifier. Alpha numeric and max size 12 chars.
		 *  @param currency ISO 4217 3 Digit Currency Code, e.g. EUR / USD / GBP
		 *  @param amount Transaction Amount, Double formatted to 2 decimal places.
		 *  @param description Transaction Description
		 *  @param email Cardholder e-mail
		 *  @param cardNumber A valid Card Number that passes the Luhn Check.
		 *  @param cardType
		 *  Card Type (Accepted Card Types must be configured in the Merchant Selfcare System.)
		 *
		 *  Accepted Values :
		 *
		 *  VISA
		 *  MASTERCARD
		 *  LASER
		 *  SWITCH
		 *  SOLO
		 *  AMEX
		 *  DINERS
		 *  MAESTRO
		 *  DELTA
		 *  ELECTRON
		 *
		 *  @param cardExpiry Card Expiry formatted MMYY
		 *  @param cardHolderName Card Holder Name
		 */
		public function XmlPreAuthCompletionRequest($terminalId,
				$orderId,
				$amount)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->terminalId = $terminalId;
				$this->orderId = $orderId;
				$this->amount = $amount;
		}
	   /**
		 *  Setter for Card Verification Value
		 *
		 *  @param cvv Numeric field with a max of 4 characters.
		 */
		public function SetCvv($cvv)
		{
				$this->cvv = $cvv;
		}
	   /**
		 *  Setter for transaction description
		 *
		 *  @param cvv Discretionary text value
		 */
		public function SetDescription($description)
		{
				$this->description = $description;
		}
	   /**
		 *  Setter for Foreign Currency Information
		 *
		 *  @param cardCurrency ISO 4217 3 Digit Currency Code, e.g. EUR / USD / GBP
		 *  @param cardAmount (Amount X Conversion rate) Formatted to two decimal places
		 *  @param conversionRate Converstion rate supplied in rate response
		 */
		public function SetForeignCurrencyInformation($cardCurrency, $cardAmount, $conversionRate)
		{
				$this->cardCurrency = $cardCurrency;
				$this->cardAmount = $cardAmount;
				$this->conversionRate = $conversionRate;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlPreAuthCompletionResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				$this->hash = $this->GetRequestHash($this->terminalId . $this->orderId . $this->amount . $this->dateTime . $sharedSecret);
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlPreAuthCompletionResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("PREAUTHCOMPLETION");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("ORDERID");
				$node->appendChild($requestXml->createTextNode($this->orderId));
				$requestString->appendChild($node);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("AMOUNT");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->amount);
				$node->appendChild($nodeText);

				if($this->cardCurrency !== NULL && $this->cardAmount > 0 && $this->conversionRate > 0)
				{
					$dcNode = $requestXml->createElement("FOREIGNCURRENCYINFORMATION");
					$requestString->appendChild($dcNode );

					$dcSubNode = $requestXml->createElement("CARDCURRENCY");
					$dcSubNode ->appendChild($requestXml->createTextNode($this->cardCurrency));
					$dcNode->appendChild($dcSubNode);

					$dcSubNode = $requestXml->createElement("CARDAMOUNT");
					$dcSubNode ->appendChild($requestXml->createTextNode($this->cardAmount));
					$dcNode->appendChild($dcSubNode);

					$dcSubNode = $requestXml->createElement("CONVERSIONRATE");
					$dcSubNode ->appendChild($requestXml->createTextNode($this->conversionRate));
					$dcNode->appendChild($dcSubNode);
				}

				if($this->description !== NULL)
				{
    				$node = $requestXml->createElement("DESCRIPTION");
    				$requestString->appendChild($node);
    				$nodeText = $requestXml->createTextNode($this->description);
    				$node->appendChild($nodeText);
                }

				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				if($this->cvv !== NULL)
				{
					$node = $requestXml->createElement("CVV");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->cvv);
					$node->appendChild($nodeText);
				}

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				return $requestXml->saveXML();

		}
}

/**
 *  Used for processing XML PreAuthorisation Completions through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlRateRequest extends Request
{
		private $terminalId;
		private $cardBin;

		/**
		 *  Creates the rate request for processing an XML Transaction
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 *  @param cardBin First 6 digits of the card number
		 */
		public function XmlRateRequest($terminalId,
				$cardBin)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->terminalId = $terminalId;
				$this->cardBin = $cardBin;
		}
			   /**
		 *  Setter for Card Verification Value
		 *
		 *  @param cvv Numeric field with a max of 4 characters.
		 */
		 private $baseAmount;
		public function SetBaseAmount($baseAmount)
		{
				$this->baseAmount = $baseAmount;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlRateResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				$this->hash = $this->GetRequestHash($this->terminalId . $this->cardBin . $this->dateTime . $sharedSecret);
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlRateResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("GETCARDCURRENCYRATE");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CARDBIN");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardBin);
				$node->appendChild($nodeText);
				
				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				if($this->baseAmount != NULL )
				{
					$node = $requestXml->createElement("BASEAMOUNT");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->baseAmount);
					$node->appendChild($nodeText);
				}

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				return $requestXml->saveXML();
		}
}

/**
 *  Used for processing XML SecureCard Registrations through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlSecureCardRegRequest extends Request
{
		private $merchantRef;
		private $terminalId;
		private $cardNumber;
		private $cardExpiry;
		private $cardHolderName;
		private $dateTime;
		private $hash;

		/**
		 *  Creates the SecureCard Registration/Update request for processing
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param merchantRef A unique card identifier. Alpha numeric and max size 48 chars.
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 *  @param cardNumber A valid Card Number that passes the Luhn Check.
		 *  @param cardType
		 *  Card Type (Accepted Card Types must be configured in the Merchant Selfcare System.)
		 *
		 *  Accepted Values :
		 *
		 *  VISA
		 *  MASTERCARD
		 *  LASER
		 *  SWITCH
		 *  SOLO
		 *  AMEX
		 *  DINERS
		 *  MAESTRO
		 *  DELTA
		 *  ELECTRON
		 *
		 *  @param cardExpiry Card Expiry formatted MMYY
		 *  @param cardHolderName Card Holder Name
		 */
		public function XmlSecureCardRegRequest($merchantRef,
				$terminalId,
				$cardNumber,
				$cardExpiry,
				$cardType,
				$cardHolderName)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->merchantRef = $merchantRef;
				$this->terminalId = $terminalId;
				$this->cardNumber = $cardNumber;
				$this->cardExpiry = $cardExpiry;
				$this->cardType = $cardType;
				$this->cardHolderName = $cardHolderName;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlSecureCardRegResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				$this->hash = $this->GetRequestHash($this->terminalId . $this->merchantRef . $this->dateTime . $this->cardNumber . $this->cardExpiry . $this->cardType . $this->cardHolderName . $sharedSecret);
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlSecureCardRegResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("SECURECARDREGISTRATION");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("MERCHANTREF");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->merchantRef);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CARDNUMBER");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardNumber);
				$node->appendChild($nodeText);
				
				$node = $requestXml->createElement("CARDEXPIRY");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardExpiry);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CARDTYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardType);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CARDHOLDERNAME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardHolderName);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				return $requestXml->saveXML();
		}
}

/**
 *  Used for processing XML SecureCard Registrations through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlSecureCardUpdRequest extends Request
{
		private $merchantRef;
		private $terminalId;
		private $cardNumber;
		private $cardExpiry;
		private $cardHolderName;
		private $dateTime;
		private $hash;

		/**
		 *  Creates the SecureCard Registration/Update request for processing
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param merchantRef A unique card identifier. Alpha numeric and max size 48 chars.
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 *  @param cardNumber A valid Card Number that passes the Luhn Check.
		 *  @param cardType
		 *  Card Type (Accepted Card Types must be configured in the Merchant Selfcare System.)
		 *
		 *  Accepted Values :
		 *
		 *  VISA
		 *  MASTERCARD
		 *  LASER
		 *  SWITCH
		 *  SOLO
		 *  AMEX
		 *  DINERS
		 *  MAESTRO
		 *  DELTA
		 *  ELECTRON
		 *
		 *  @param cardExpiry Card Expiry formatted MMYY
		 *  @param cardHolderName Card Holder Name
		 */
		public function XmlSecureCardUpdRequest($merchantRef,
				$terminalId,
				$cardNumber,
				$cardExpiry,
				$cardType,
				$cardHolderName)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->merchantRef = $merchantRef;
				$this->terminalId = $terminalId;
				$this->cardNumber = $cardNumber;
				$this->cardExpiry = $cardExpiry;
				$this->cardType = $cardType;
				$this->cardHolderName = $cardHolderName;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlSecureCardRegResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				$this->hash = $this->GetRequestHash($this->terminalId . $this->merchantRef . $this->dateTime . $this->cardNumber . $this->cardExpiry . $this->cardType . $this->cardHolderName . $sharedSecret);
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlSecureCardUpdResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("SECURECARDUPDATE");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("MERCHANTREF");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->merchantRef);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CARDNUMBER");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardNumber);
				$node->appendChild($nodeText);
				
				$node = $requestXml->createElement("CARDEXPIRY");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardExpiry);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CARDTYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardType);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("CARDHOLDERNAME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->cardHolderName);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				return $requestXml->saveXML();
		}
}
/**
 *  Used for processing XML SecureCard searching through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlSecureCardSearchRequest extends Request
{
		private $merchantRef;
		private $terminalId;
		private $dateTime;
		private $hash;

		/**
		 *  Creates the SecureCard searche request for processing
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param merchantRef A unique card identifier. Alpha numeric and max size 48 chars.
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 */
		public function XmlSecureCardSearchRequest($merchantRef,
				$terminalId)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->merchantRef = $merchantRef;
				$this->terminalId = $terminalId;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlSecureCardRegResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				$this->hash = $this->GetRequestHash($this->terminalId . $this->merchantRef . $this->dateTime . $sharedSecret);
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlSecureCardSearchResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("SECURECARDSEARCH");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("MERCHANTREF");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->merchantRef);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				return $requestXml->saveXML();
		}
}

/**
 *  Used for processing XML Stored Subscription Registrations through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlStoredSubscriptionRegRequest extends Request
{
		private $merchantRef;
		private $terminalId;
		private $name;
		private $description;
		private $periodType;
		private $length;
		private $recurringAmount;
		private $initialAmount;
		private $type;
		private $onUpdate;
		private $onDelete;
		private $dateTime;
		private $hash;

		/**
		 *  Creates the SecureCard Registration/Update request for processing
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param merchantRef A unique subscription identifier. Alpha numeric and max size 48 chars.
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 *  @param secureCardMerchantRef A valid, registered SecureCard Merchant Reference.
		 *  @param name Name of the subscription
		 *  @param description Card Holder Name
		 */
		public function XmlStoredSubscriptionRegRequest($merchantRef,
				$terminalId,
				$startDate,
				$name,
				$description,
				$periodType,
				$length,
				$recurringAmount,
				$initialAmount,
				$type,
				$onUpdate,
				$onDelete)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->merchantRef = $merchantRef;
				$this->terminalId = $terminalId;
				$this->startDate = $startDate;

				$this->name = $name;
				$this->description = $description;
				$this->periodType = $periodType;
				$this->length = $length;
				$this->recurringAmount = $recurringAmount;
				$this->initialAmount = $initialAmount;
				$this->type = $type;
				$this->onUpdate = $onUpdate;
				$this->onDelete = $onDelete;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlSecureCardRegResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				$this->hash = $this->GetRequestHash($this->terminalId . $this->merchantRef . $this->dateTime . $this->type . $this->name . $this->periodType . $this->recurringAmount . $this->initialAmount . $this->length . $sharedSecret);
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlStoredSubscriptionRegResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("ADDSTOREDSUBSCRIPTION");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("MERCHANTREF");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->merchantRef);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("NAME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->name);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DESCRIPTION");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->description);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("PERIODTYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->periodType);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("LENGTH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->length);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("RECURRINGAMOUNT");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->recurringAmount);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("INITIALAMOUNT");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->initialAmount);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->type);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("ONUPDATE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->onUpdate);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("ONDELETE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->onDelete);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				return $requestXml->saveXML();
		}
}
/**
 *  Used for processing XML Stored Subscription Registrations through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlStoredSubscriptionUpdRequest extends Request
{
		private $merchantRef;
		private $terminalId;
		private $name;
		private $description;
		private $periodType;
		private $length;
		private $recurringAmount;
		private $initialAmount;
		private $type;
		private $onUpdate;
		private $onDelete;
		private $dateTime;
		private $hash;

		/**
		 *  Creates the SecureCard Registration/Update request for processing
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param merchantRef A unique subscription identifier. Alpha numeric and max size 48 chars.
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 *  @param secureCardMerchantRef A valid, registered SecureCard Merchant Reference.
		 *  @param name Name of the subscription
		 *  @param description Card Holder Name
		 */
		public function XmlStoredSubscriptionUpdRequest($merchantRef,
				$terminalId,
				$startDate,
				$name,
				$description,
				$periodType,
				$length,
				$recurringAmount,
				$initialAmount,
				$type,
				$onUpdate,
				$onDelete)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->merchantRef = $merchantRef;
				$this->terminalId = $terminalId;
				$this->startDate = $startDate;

				$this->name = $name;
				$this->description = $description;
				$this->periodType = $periodType;
				$this->length = $length;
				$this->recurringAmount = $recurringAmount;
				$this->initialAmount = $initialAmount;
				$this->type = $type;
				$this->onUpdate = $onUpdate;
				$this->onDelete = $onDelete;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlSecureCardRegResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				$this->hash = $this->GetRequestHash($this->terminalId . $this->merchantRef . $this->dateTime . $this->type . $this->name . $this->periodType . $this->recurringAmount . $this->initialAmount . $this->length . $sharedSecret);
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlStoredSubscriptionUpdResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("UPDATESTOREDSUBSCRIPTION");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("MERCHANTREF");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->merchantRef);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("NAME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->name);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DESCRIPTION");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->description);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("PERIODTYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->periodType);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("LENGTH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->length);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("RECURRINGAMOUNT");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->recurringAmount);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("INITIALAMOUNT");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->initialAmount);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->type);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("ONUPDATE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->onUpdate);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("ONDELETE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->onDelete);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				return $requestXml->saveXML();
		}
}

/**
 *  Used for processing XML SecureCard Registrations through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlStoredSubscriptionDelRequest extends Request
{
		private $merchantRef;
		private $terminalId;
		private $dateTime;
		private $hash;

		/**
		 *  Creates the SecureCard Registration/Update request for processing
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param merchantRef A unique subscription identifier. Alpha numeric and max size 48 chars.
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 */
		public function XmlStoredSubscriptionDelRequest($merchantRef,
				$terminalId)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->merchantRef = $merchantRef;
				$this->terminalId = $terminalId;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlSecureCardRegResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				$this->hash = $this->GetRequestHash($this->terminalId . $this->merchantRef . $this->dateTime . $sharedSecret);
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlStoredSubscriptionDelResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("DELETESTOREDSUBSCRIPTION");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("MERCHANTREF");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->merchantRef);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				return $requestXml->saveXML();
		}
}

/**
 *  Used for processing XML SecureCard Registrations through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlSubscriptionRegRequest extends Request
{
		private $merchantRef;
		private $terminalId;
		private $storedSubscriptionRef;
		private $secureCardMerchantRef;
		private $name;
		private $description;
		private $periodType;
		private $length;
		private $recurringAmount;
		private $initialAmount;
		private $type;
		private $startDate;
		private $endDate;
		private $onUpdate;
		private $onDelete;
		private $dateTime;
		private $hash;

		private $newStoredSubscription = false;

		/**
		 *  Creates the SecureCard Registration/Update request for processing
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param merchantRef A unique subscription identifier. Alpha numeric and max size 48 chars.
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 *  @param secureCardMerchantRef A valid, registered SecureCard Merchant Reference.
		 *  @param name Name of the subscription
		 *  @param description Card Holder Name
		 */
		public function SetNewStoredSubscriptionValues($name,
				$description,
				$periodType,
				$recurringAmount,
				$initialAmount,
				$type,
				$onUpdate,
				$onDelete)
		{
				$this->name = $name;
				$this->description = $description;
				$this->periodType = $periodType;
				$this->recurringAmount = $recurringAmount;
				$this->initialAmount = $initialAmount;
				$this->type = $type;
				$this->onUpdate = $onUpdate;
				$this->onDelete = $onDelete;

				$this->newStoredSubscription = true;
		}
	   /**
		 *  Setter for stored subscription reference
		 *
		 *  @param endDate Stored subscription reference
		 */
		public function SetEndDate($endDate)
		{
				$this->endDate = $endDate;
		}
	   /**
		 *  Setter for stored subscription length
		 *
		 *  @param length Stored subscription length
		 */
		public function SetLength($length)
		{
				$this->length = $length;
		}
		/**
		 *  Creates the SecureCard Registration/Update request for processing
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param merchantRef A unique subscription identifier. Alpha numeric and max size 48 chars.
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 *  @param secureCardMerchantRef A valid, registered SecureCard Merchant Reference.
		 *  @param name Name of the subscription
		 *  @param description Card Holder Name
		 */
		public function XmlSubscriptionRegRequest($merchantRef,
				$terminalId,
				$storedSubscriptionRef,
				$secureCardMerchantRef,
				$startDate)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->storedSubscriptionRef = $storedSubscriptionRef;
				$this->secureCardMerchantRef = $secureCardMerchantRef;
				$this->merchantRef = $merchantRef;
				$this->terminalId = $terminalId;
				$this->startDate = $startDate;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlSecureCardRegResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				$this->hash = $this->GetRequestHash($this->terminalId . $this->merchantRef . $this->storedSubscriptionRef . $this->secureCardMerchantRef . $this->dateTime . $this->startDate . $sharedSecret);
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlSubscriptionRegResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("ADDSUBSCRIPTION");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("MERCHANTREF");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->merchantRef);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("STOREDSUBSCRIPTIONREF");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->storedSubscriptionRef);
				$node->appendChild($nodeText);
					
				$node = $requestXml->createElement("SECURECARDMERCHANTREF");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->secureCardMerchantRef);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("STARTDATE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->startDate);
				$node->appendChild($nodeText);

				if($this->newStoredSubscription)
				{
					$node = $requestXml->createElement("NAME");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->name);
					$node->appendChild($nodeText);

					$node = $requestXml->createElement("DESCRIPTION");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->description);
					$node->appendChild($nodeText);

					$node = $requestXml->createElement("PERIODTYPE");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->periodType);
					$node->appendChild($nodeText);

					if($this->length != null)
					{
						$node = $requestXml->createElement("LENGTH");
						$requestString->appendChild($node);
						$nodeText = $requestXml->createTextNode($this->length);
						$node->appendChild($nodeText);
					}

					$node = $requestXml->createElement("RECURRINGAMOUNT");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->recurringAmount);
					$node->appendChild($nodeText);

					$node = $requestXml->createElement("INITIALAMOUNT");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->initialAmount);
					$node->appendChild($nodeText);

					$node = $requestXml->createElement("TYPE");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->type);
					$node->appendChild($nodeText);

					if($this->endDate != null)
					{
						$node = $requestXml->createElement("ENDDATE");
						$requestString->appendChild($node);
						$nodeText = $requestXml->createTextNode($this->endDate);
						$node->appendChild($nodeText);
					}

					$node = $requestXml->createElement("ONUPDATE");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->onUpdate);
					$node->appendChild($nodeText);

					$node = $requestXml->createElement("ONDELETE");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->onDelete);
					$node->appendChild($nodeText);
				}

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				return $requestXml->saveXML();
		}
}

/**
 *  Used for processing XML SecureCard Registrations through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlSubscriptionUpdRequest extends Request
{
		private $merchantRef;
		private $terminalId;
		private $storedSubscriptionRef;
		private $secureCardMerchantRef;
		private $name;
		private $description;
		private $periodType;
		private $length;
		private $recurringAmount;
		private $initialAmount;
		private $type;
		private $startDate;
		private $endDate;
		private $onUpdate;
		private $onDelete;
		private $dateTime;
		private $hash;

	   /**
		 *  Setter for stored subscription reference
		 *
		 *  @param endDate Stored subscription reference
		 */
		public function SetEndDate($endDate)
		{
				$this->endDate = $endDate;
		}
	   /**
		 *  Setter for stored subscription length
		 *
		 *  @param length Stored subscription length
		 */
		public function SetLength($length)
		{
				$this->length = $length;
		}
		/**
		 *  Creates the SecureCard Registration/Update request for processing
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param merchantRef A unique subscription identifier. Alpha numeric and max size 48 chars.
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 *  @param secureCardMerchantRef A valid, registered SecureCard Merchant Reference.
		 *  @param name Name of the subscription
		 *  @param description Card Holder Name
		 */
		public function XmlSubscriptionUpdRequest($merchantRef,
				$terminalId,
				$secureCardMerchantRef,
				$startDate,
				$name,
				$description,
				$periodType,
				$recurringAmount,
				$initialAmount,
				$type,
				$onUpdate,
				$onDelete)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->secureCardMerchantRef = $secureCardMerchantRef;
				$this->merchantRef = $merchantRef;
				$this->terminalId = $terminalId;
				$this->startDate = $startDate;

				$this->name = $name;
				$this->description = $description;
				$this->periodType = $periodType;
				$this->recurringAmount = $recurringAmount;
				$this->initialAmount = $initialAmount;
				$this->type = $type;
				$this->onUpdate = $onUpdate;
				$this->onDelete = $onDelete;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlSecureCardRegResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				$this->hash = $this->GetRequestHash($this->terminalId . $this->merchantRef . $this->secureCardMerchantRef . $this->dateTime . $this->startDate . $sharedSecret);
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlSubscriptionUpdResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("UPDATESUBSCRIPTION");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("MERCHANTREF");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->merchantRef);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("SECURECARDMERCHANTREF");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->secureCardMerchantRef);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("STARTDATE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->startDate);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("NAME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->name);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DESCRIPTION");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->description);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("PERIODTYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->periodType);
				$node->appendChild($nodeText);

				if($this->length != null)
				{
					$node = $requestXml->createElement("LENGTH");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->length);
					$node->appendChild($nodeText);
				}

				$node = $requestXml->createElement("RECURRINGAMOUNT");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->recurringAmount);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("INITIALAMOUNT");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->initialAmount);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TYPE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->type);
				$node->appendChild($nodeText);

				if($this->endDate != null)
				{
					$node = $requestXml->createElement("ENDDATE");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->endDate);
					$node->appendChild($nodeText);
				}

				$node = $requestXml->createElement("ONUPDATE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->onUpdate);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("ONDELETE");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->onDelete);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				return $requestXml->saveXML();
		}
}

/**
 *  Used for processing XML SecureCard Registrations through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlSubscriptionDelRequest extends Request
{
		private $merchantRef;
		private $terminalId;
		private $dateTime;
		private $hash;

		/**
		 *  Creates the SecureCard Registration/Update request for processing
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param merchantRef A unique subscription identifier. Alpha numeric and max size 48 chars.
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 */
		public function XmlSubscriptionDelRequest($merchantRef,
				$terminalId)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->merchantRef = $merchantRef;
				$this->terminalId = $terminalId;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlSecureCardRegResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				$this->hash = $this->GetRequestHash($this->terminalId . $this->merchantRef . $this->dateTime . $sharedSecret);
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlSubscriptionDelResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("DELETESUBSCRIPTION");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("MERCHANTREF");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->merchantRef);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				return $requestXml->saveXML();
		}
}

/**
 *  Used for processing XML SecureCard Registrations through the WorldNet TPS XML Gateway.
 *
 *  Basic request is configured on initialisation and optional fields can be configured.
 */
class XmlSubscriptionPaymentRequest extends Request
{
		private $merchantRef;
		private $terminalId;
		private $orderId;
		private $amount;
		private $subscriptionRef;
		private $cardCurrency;
		private $cardAmount;
		private $conversionRate;
		private $email;
		private $dateTime;
		private $hash;

	   /**
		 *  Setter for Email Address Value
		 *
		 *  @param email Alpha-numeric field.
		 */
		public function SetEmail($email)
		{
				$this->email = $email;
		}
	   /**
		 *  Setter for Foreign Currency Information
		 *
		 *  @param cardCurrency ISO 4217 3 Digit Currency Code, e.g. EUR / USD / GBP
		 *  @param cardAmount (Amount X Conversion rate) Formatted to two decimal places
		 *  @param conversionRate Converstion rate supplied in rate response
		 */
		public function SetForeignCurrencyInformation($cardCurrency, $cardAmount, $conversionRate)
		{
				$this->cardCurrency = $cardCurrency;
				$this->cardAmount = $cardAmount;
				$this->conversionRate = $conversionRate;
		}
		/**
		 *  Creates the SecureCard Registration/Update request for processing
		 *  through the WorldNetTPS XML Gateway
		 *
		 *  @param merchantRef A unique subscription identifier. Alpha numeric and max size 48 chars.
		 *  @param terminalId Terminal ID provided by WorldNet TPS
		 */
		public function XmlSubscriptionPaymentRequest($merchantRef,
				$terminalId,
				$amount,
				$subscriptionRef)
		{
				$this->dateTime = $this->GetFormattedDate();

				$this->merchantRef = $merchantRef;
				$this->terminalId = $terminalId;
				$this->amount = $amount;
				$this->subscriptionRef = $subscriptionRef;
		}
	   /**
		 *  Method to process transaction and return parsed response from the WorldNet TPS XML Gateway
		 *
		 *  @param sharedSecret
		 *  Shared secret either supplied by WorldNet TPS or configured under
		 *  Terminal Settings in the Merchant Selfcare System.
		 *
		 *  @param testMode
		 *  Boolean value defining Mode
		 *  true - Test mode active
		 *  false - Production mode, all transactions will be processed by Issuer.
		 *
		 *  @return XmlSecureCardRegResponse containing an error or the parsed payment response.
		 */
		public function ProcessRequest($sharedSecret, $testMode)
		{
				$this->hash = $this->GetRequestHash($this->terminalId . $this->merchantRef . $this->dateTime . $sharedSecret);
				$responseString = $this->SendRequest($this->GenerateXml(), $testMode);
				$response = new XmlSubscriptionPaymentResponse($responseString);
				return $response;
		}

		private function GenerateXml()
		{
				$requestXml = new DOMDocument("1.0");
				$requestXml->formatOutput = true;

				$requestString = $requestXml->createElement("SUBSCRIPTIONPAYMENT");
				$requestXml->appendChild($requestString);

				$node = $requestXml->createElement("ORDERID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->orderId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("TERMINALID");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->terminalId);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("AMOUNT");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->amount);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("SUBSCRIPTIONREF");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->subscriptionRef);
				$node->appendChild($nodeText);

				if($this->cardCurrency !== NULL && $this->cardAmount > 0 && $this->conversionRate > 0)
				{
					$dcNode = $requestXml->createElement("FOREIGNCURRENCYINFORMATION");
					$requestString->appendChild($dcNode );

					$dcSubNode = $requestXml->createElement("CARDCURRENCY");
					$dcSubNode ->appendChild($requestXml->createTextNode($this->cardCurrency));
					$dcNode->appendChild($dcSubNode);

					$dcSubNode = $requestXml->createElement("CARDAMOUNT");
					$dcSubNode ->appendChild($requestXml->createTextNode($this->cardAmount));
					$dcNode->appendChild($dcSubNode);

					$dcSubNode = $requestXml->createElement("CONVERSIONRATE");
					$dcSubNode ->appendChild($requestXml->createTextNode($this->conversionRate));
					$dcNode->appendChild($dcSubNode);
				}

				if($this->email !== NULL)
				{
					$node = $requestXml->createElement("EMAIL");
					$requestString->appendChild($node);
					$nodeText = $requestXml->createTextNode($this->email);
					$node->appendChild($nodeText);
				}

				$node = $requestXml->createElement("DATETIME");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->dateTime);
				$node->appendChild($nodeText);

				$node = $requestXml->createElement("HASH");
				$requestString->appendChild($node);
				$nodeText = $requestXml->createTextNode($this->hash);
				$node->appendChild($nodeText);

				return $requestXml->saveXML();
		}
}

/**
  *  Holder class for parsed response. If there was an error there will be an error string 
  *  otherwise all values will be populated with the parsed payment response values.
  *  
  *  IsError should be checked before accessing any fields.
  *  
  *  ErrorString will contain the error if one occurred.
  */
class XmlAuthResponse
{
		private $isError = false;
		public function IsError()
		{
				return $this->isError;
		}
		
		private $errorString;
		public function ErrorString()
		{
				return $this->errorString;
		}

		private $responseCode;
		public function ResponseCode()
		{
				return $this->responseCode;
		}
		
		private $responseText;
		public function ResponseText()
		{
				return $this->responseText;
		}
		
		private $approvalCode;
		public function ApprovalCode()
		{
				return $this->approvalCode;
		}
		
		private $dateTime;
		public function DateTime()
		{
				return $this->dateTime;
		}
		
		private $avsResponse;
		public function AvsResponse()
		{
				return $this->avsResponse;
		}
		
		private $cvvResponse;
		public function CvvResponse()
		{
				return $this->cvvResponse;
		}
		
		private $hash;
		public function Hash()
		{
				return $this->hash;
		}

		public function XmlAuthResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;	
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "PAYMENTRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("PAYMENTRESPONSE");
	
								foreach( $responseNodes as $node )
								{
									$this->responseCode = $node->getElementsByTagName('RESPONSECODE')->item(0)->nodeValue;
									$this->responseText = $node->getElementsByTagName('RESPONSETEXT')->item(0)->nodeValue;
									$this->approvalCode = $node->getElementsByTagName('APPROVALCODE')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->avsResponse = $node->getElementsByTagName('AVSRESPONSE')->item(0)->nodeValue;
									$this->cvvResponse = $node->getElementsByTagName('CVVRESPONSE')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;		
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  Holder class for parsed response. If there was an error there will be an error string
  *  otherwise all values will be populated with the parsed payment response values.
  *
  *  IsError should be checked before accessing any fields.
  *
  *  ErrorString will contain the error if one occurred.
  */
class XmlRefundResponse
{
		private $isError = false;
		public function IsError()
		{
				return $this->isError;
		}
		
		private $errorString;
		public function ErrorString()
		{
				return $this->errorString;
		}

		private $responseCode;
		public function ResponseCode()
		{
				return $this->responseCode;
		}
		
		private $responseText;
		public function ResponseText()
		{
				return $this->responseText;
		}
		
		private $approvalCode;
		public function OrderId()
		{
				return $this->orderId;
		}
		
		private $avsResponse;
		public function TerminalId()
		{
				return $this->terminalId;
		}
		
		private $cvvResponse;
		public function Amount()
		{
				return $this->amount;
		}
		
		private $dateTime;
		public function DateTime()
		{
				return $this->dateTime;
		}
		
		private $hash;
		public function Hash()
		{
				return $this->hash;
		}

		public function XmlRefundResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;	
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "REFUNDRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("REFUNDRESPONSE");
	
								foreach( $responseNodes as $node )
								{
									$this->responseCode = $node->getElementsByTagName('RESPONSECODE')->item(0)->nodeValue;
									$this->responseText = $node->getElementsByTagName('RESPONSETEXT')->item(0)->nodeValue;
									$this->orderId = $node->getElementsByTagName('ORDERID')->item(0)->nodeValue;
									$this->terminalId = $node->getElementsByTagName('TERMINALID')->item(0)->nodeValue;
									$this->amount = $node->getElementsByTagName('AMOUNT')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;		
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  Holder class for parsed response. If there was an error there will be an error string
  *  otherwise all values will be populated with the parsed payment response values.
  *
  *  IsError should be checked before accessing any fields.
  *
  *  ErrorString will contain the error if one occurred.
  */
class XmlPreAuthResponse
{
		private $isError = false;
		public function IsError()
		{
				return $this->isError;
		}

		private $errorString;
		public function ErrorString()
		{
				return $this->errorString;
		}

		private $responseCode;
		public function ResponseCode()
		{
				return $this->responseCode;
		}

		private $responseText;
		public function ResponseText()
		{
				return $this->responseText;
		}

		private $approvalCode;
		public function ApprovalCode()
		{
				return $this->approvalCode;
		}

		private $dateTime;
		public function DateTime()
		{
				return $this->dateTime;
		}

		private $avsResponse;
		public function AvsResponse()
		{
				return $this->avsResponse;
		}

		private $cvvResponse;
		public function CvvResponse()
		{
				return $this->cvvResponse;
		}

		private $hash;
		public function Hash()
		{
				return $this->hash;
		}

		public function XmlPreAuthResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "PREAUTHRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("PREAUTHRESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->responseCode = $node->getElementsByTagName('RESPONSECODE')->item(0)->nodeValue;
									$this->responseText = $node->getElementsByTagName('RESPONSETEXT')->item(0)->nodeValue;
									$this->approvalCode = $node->getElementsByTagName('APPROVALCODE')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->avsResponse = $node->getElementsByTagName('AVSRESPONSE')->item(0)->nodeValue;
									$this->cvvResponse = $node->getElementsByTagName('CVVRESPONSE')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  Holder class for parsed response. If there was an error there will be an error string
  *  otherwise all values will be populated with the parsed payment response values.
  *
  *  IsError should be checked before accessing any fields.
  *
  *  ErrorString will contain the error if one occurred.
  */
class XmlPreAuthCompletionResponse
{
		private $isError = false;
		public function IsError()
		{
				return $this->isError;
		}

		private $errorString;
		public function ErrorString()
		{
				return $this->errorString;
		}

		private $responseCode;
		public function ResponseCode()
		{
				return $this->responseCode;
		}

		private $responseText;
		public function ResponseText()
		{
				return $this->responseText;
		}

		private $approvalCode;
		public function ApprovalCode()
		{
				return $this->approvalCode;
		}

		private $dateTime;
		public function DateTime()
		{
				return $this->dateTime;
		}

		private $avsResponse;
		public function AvsResponse()
		{
				return $this->avsResponse;
		}

		private $cvvResponse;
		public function CvvResponse()
		{
				return $this->cvvResponse;
		}

		private $hash;
		public function Hash()
		{
				return $this->hash;
		}

		public function XmlPreAuthCompletionResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "PREAUTHCOMPLETIONRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("PREAUTHCOMPLETIONRESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->responseCode = $node->getElementsByTagName('RESPONSECODE')->item(0)->nodeValue;
									$this->responseText = $node->getElementsByTagName('RESPONSETEXT')->item(0)->nodeValue;
									$this->approvalCode = $node->getElementsByTagName('APPROVALCODE')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->avsResponse = $node->getElementsByTagName('AVSRESPONSE')->item(0)->nodeValue;
									$this->cvvResponse = $node->getElementsByTagName('CVVRESPONSE')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  Holder class for parsed response. If there was an error there will be an error string
  *  otherwise all values will be populated with the parsed payment response values.
  *
  *  IsError should be checked before accessing any fields.
  *
  *  ErrorString will contain the error if one occurred.
  */
class XmlRateResponse
{
		private $isError = false;
		public function IsError()
		{
				return $this->isError;
		}

		private $errorString;
		public function ErrorString()
		{
				return $this->errorString;
		}

		private $terminalCurrency;
		public function TerminalCurrency()
		{
				return $this->terminalCurrency;
		}

		private $cardCurrency;
		public function CardCurrency()
		{
				return $this->cardCurrency;
		}

		private $conversionRate;
		public function ConversionRate()
		{
				return $this->conversionRate;
		}
		private $foreignAmount;
		public function ForeignAmount()
		{
				return $this->foreignAmount;
		}

		private $dateTime;
		public function DateTime()
		{
				return $this->dateTime;
		}

		private $hash;
		public function Hash()
		{
				return $this->hash;
		}

		public function XmlRateResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "CARDCURRENCYRATERESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("CARDCURRENCYRATERESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->terminalCurrency = $node->getElementsByTagName('TERMINALCURRENCY')->item(0)->nodeValue;
									$this->cardCurrency = $node->getElementsByTagName('CARDCURRENCY')->item(0)->nodeValue;
									$this->conversionRate = $node->getElementsByTagName('CONVERSIONRATE')->item(0)->nodeValue;
									$this->foreignAmount = $node->getElementsByTagName('FOREIGNAMOUNT')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  Base holder class for parsed SecureCard response. If there was an error there will be an error string
  *  otherwise all values will be populated with the parsed payment response values.
  */
class XmlSecureCardResponse
{
		protected $isError = false;
		public function IsError()
		{
				return $this->isError;
		}

		protected $errorString;
		public function ErrorString()
		{
				return $this->errorString;
		}

		protected $errorCode;
		public function ErrorCode()
		{
				return $this->errorCode;
		}

		protected $merchantRef;
		public function MerchantReference()
		{
				return $this->merchantRef;
		}

		protected $cardRef;
		public function CardReference()
		{
				return $this->cardRef;
		}

		protected $dateTime;
		public function DateTime()
		{
				return $this->dateTime;
		}

		protected $hash;
		public function Hash()
		{
				return $this->hash;
		}
}

/**
  *  Holder class for parsed SecureCard registration response. 
  */
class XmlSecureCardRegResponse extends XmlSecureCardResponse
{
		public function XmlSecureCardRegResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorCode = $node->getElementsByTagName('ERRORCODE')->item(0)->nodeValue;
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "SECURECARDREGISTRATIONRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("SECURECARDREGISTRATIONRESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->merchantRef = $node->getElementsByTagName('MERCHANTREF')->item(0)->nodeValue;
									$this->cardRef = $node->getElementsByTagName('CARDREFERENCE')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  Holder class for parsed SecureCard update response. 
  */
class XmlSecureCardUpdResponse extends XmlSecureCardResponse
{
		public function XmlSecureCardUpdResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorCode = $node->getElementsByTagName('ERRORCODE')->item(0)->nodeValue;
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "SECURECARDUPDATERESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("SECURECARDUPDATERESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->merchantRef = $node->getElementsByTagName('MERCHANTREF')->item(0)->nodeValue;
									$this->cardRef = $node->getElementsByTagName('CARDREFERENCE')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  Holder class for parsed SecureCard search response. 
  */
class XmlSecureCardSearchResponse extends XmlSecureCardResponse
{
		private $cardType;
		public function CardType()
		{
				return $this->cardType;
		}

		private $expiry;
		public function CardExpiry()
		{
				return $this->expiry;
		}

		private $cardHolderName;
		public function CardHolderName()
		{
				return $this->cardHolderName;
		}

		public function XmlSecureCardSearchResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorCode = $node->getElementsByTagName('ERRORCODE')->item(0)->nodeValue;
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "SECURECARDSEARCHRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("SECURECARDSEARCHRESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->merchantRef = $node->getElementsByTagName('MERCHANTREF')->item(0)->nodeValue;
									$this->cardRef = $node->getElementsByTagName('CARDREFERENCE')->item(0)->nodeValue;
									$this->cardType = $node->getElementsByTagName('CARDTYPE')->item(0)->nodeValue;
									$this->expiry = $node->getElementsByTagName('CARDEXPIRY')->item(0)->nodeValue;
									$this->cardHolderName = $node->getElementsByTagName('CARDHOLDERNAME')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  Base holder class for parsed Subscription response. If there was an error there will be an error string
  *  otherwise all values will be populated with the parsed payment response values.
  */
class XmlSubscriptionResponse
{
		protected $isError = false;
		public function IsError()
		{
				return $this->isError;
		}

		protected $errorString;
		public function ErrorString()
		{
				return $this->errorString;
		}

		protected $errorCode;
		public function ErrorCode()
		{
				return $this->errorCode;
		}

		protected $merchantRef;
		public function MerchantReference()
		{
				return $this->merchantRef;
		}

		protected $dateTime;
		public function DateTime()
		{
				return $this->dateTime;
		}

		protected $hash;
		public function Hash()
		{
				return $this->hash;
		}
}

/**
  *  Holder class for parsed Stored Subscription registration response. 
  */
class XmlStoredSubscriptionRegResponse extends XmlSubscriptionResponse
{
		public function XmlStoredSubscriptionRegResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorCode = $node->getElementsByTagName('ERRORCODE')->item(0)->nodeValue;
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "ADDSTOREDSUBSCRIPTIONRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("ADDSTOREDSUBSCRIPTIONRESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->merchantRef = $node->getElementsByTagName('MERCHANTREF')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  Holder class for parsed Stored Subscription update response. 
  */
class XmlStoredSubscriptionUpdResponse extends XmlSubscriptionResponse
{
		public function XmlStoredSubscriptionUpdResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorCode = $node->getElementsByTagName('ERRORCODE')->item(0)->nodeValue;
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "UPDATESTOREDSUBSCRIPTIONRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("UPDATESTOREDSUBSCRIPTIONRESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->merchantRef = $node->getElementsByTagName('MERCHANTREF')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  Holder class for parsed Stored Subscription deletion response. 
  */
class XmlStoredSubscriptionDelResponse extends XmlSubscriptionResponse
{
		public function XmlStoredSubscriptionDelResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorCode = $node->getElementsByTagName('ERRORCODE')->item(0)->nodeValue;
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "DELETESTOREDSUBSCRIPTIONRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("DELETESTOREDSUBSCRIPTIONRESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->merchantRef = $node->getElementsByTagName('MERCHANTREF')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  Holder class for parsed Subscription registration response. 
  */
class XmlSubscriptionRegResponse extends XmlSubscriptionResponse
{
		public function XmlSubscriptionRegResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorCode = $node->getElementsByTagName('ERRORCODE')->item(0)->nodeValue;
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "ADDSUBSCRIPTIONRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("ADDSUBSCRIPTIONRESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->merchantRef = $node->getElementsByTagName('MERCHANTREF')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  Holder class for parsed Subscription update response. 
  */
class XmlSubscriptionUpdResponse extends XmlSubscriptionResponse
{
		public function XmlSubscriptionUpdResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorCode = $node->getElementsByTagName('ERRORCODE')->item(0)->nodeValue;
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "UPDATESUBSCRIPTIONRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("UPDATESUBSCRIPTIONRESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->merchantRef = $node->getElementsByTagName('MERCHANTREF')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  Holder class for parsed Subscription deletion response. 
  */
class XmlSubscriptionDelResponse extends XmlSubscriptionResponse
{
		public function XmlSubscriptionDelResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorCode = $node->getElementsByTagName('ERRORCODE')->item(0)->nodeValue;
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "DELETESUBSCRIPTIONRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("DELETESUBSCRIPTIONRESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->merchantRef = $node->getElementsByTagName('MERCHANTREF')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}
/**
  *  Holder class for parsed Subscription deletion response. 
  */
class XmlSubscriptionPaymentResponse
{
		private $isError = false;
		public function IsError()
		{
				return $this->isError;
		}
		
		private $errorString;
		public function ErrorString()
		{
				return $this->errorString;
		}

		private $responseCode;
		public function ResponseCode()
		{
				return $this->responseCode;
		}
		
		private $responseText;
		public function ResponseText()
		{
				return $this->responseText;
		}
		
		private $approvalCode;
		public function ApprovalCode()
		{
				return $this->approvalCode;
		}
		
		private $dateTime;
		public function DateTime()
		{
				return $this->dateTime;
		}
		
		private $hash;
		public function Hash()
		{
				return $this->hash;
		}

		public function XmlSubscriptionPaymentResponse($responseXml)
		{
				$doc = new DOMDocument();
				$doc->loadXML($responseXml);
				try
				{
						if (strpos($responseXml, "ERROR"))
						{
								$responseNodes = $doc->getElementsByTagName("ERROR");
								foreach( $responseNodes as $node )
								{
									$this->errorCode = $node->getElementsByTagName('ERRORCODE')->item(0)->nodeValue;
									$this->errorString = $node->getElementsByTagName('ERRORSTRING')->item(0)->nodeValue;
								}
								$this->isError = true;
						}
						else if (strpos($responseXml, "SUBSCRIPTIONPAYMENTRESPONSE"))
						{
								$responseNodes = $doc->getElementsByTagName("SUBSCRIPTIONPAYMENTRESPONSE");

								foreach( $responseNodes as $node )
								{
									$this->responseCode = $node->getElementsByTagName('RESPONSECODE')->item(0)->nodeValue;
									$this->responseText = $node->getElementsByTagName('RESPONSETEXT')->item(0)->nodeValue;
									$this->approvalCode = $node->getElementsByTagName('APPROVALCODE')->item(0)->nodeValue;
									$this->dateTime = $node->getElementsByTagName('DATETIME')->item(0)->nodeValue;
									$this->hash = $node->getElementsByTagName('HASH')->item(0)->nodeValue;
								}
						}
						else
						{
								throw new Exception("Invalid Response");
						}
				}
				catch (Exception $e)
				{
						$this->isError = true;
						$this->errorString = $e->getMessage();
				}
		}
}

/**
  *  For backward compatibility with older class names.
  */
class XmlStandardRequest extends XmlAuthRequest { }
class XmlStandardResponse extends XmlAuthResponse { }

?>
